
#include <ctype.h>
#include <string.h>
#include "config.h"
#include "playback.h"
#include "infrared.h"
#include "ir_codes.h"


extern int strncasecmp(const char* s1, const char* s2, size_t n);

static unsigned char this_token_is(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short toklen = strlen(token);
	if( *buf_len-*buf_pos < toklen || strncasecmp(buf+*buf_pos, token, toklen) )
		return 0;
	*buf_pos += toklen;
	return 1;
}
static unsigned char skip_space(char* buf, unsigned short* buf_pos, unsigned short* buf_len) {
	if( *buf_pos == *buf_len || !isspace(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isspace(buf[*buf_pos]) && buf[*buf_pos] != '\n' )
		++*buf_pos;
	return 1;
}
static unsigned char next_token_is(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short pos;
	if( !isspace(buf[*buf_pos]) )
		return 0;
	pos = *buf_pos;
	skip_space(buf, &pos, buf_len);
	if( this_token_is(buf, &pos, buf_len, token) ) {
		*buf_pos = pos;
		return 1;
	} else {
		return 0;
    }   
}
static unsigned char next_token_is_spaceopt(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short pos;
	pos = *buf_pos;
	skip_space(buf, &pos, buf_len);
	if( this_token_is(buf, &pos, buf_len, token) ) {
		*buf_pos = pos;
		return 1;
	} else {
		return 0;
    }   
}

static unsigned char get_next_token_boolean(char* buf, unsigned short* buf_pos, unsigned short* buf_len) {
	if( next_token_is(buf, buf_pos, buf_len, "yes") || next_token_is(buf, buf_pos, buf_len, "true") )
		return 1;
	else if( next_token_is(buf, buf_pos, buf_len, "no") || next_token_is(buf, buf_pos, buf_len, "false") )
		return 0;
	else
		return 0xFF;
}

static unsigned char read_integer(char* buf, unsigned short* buf_pos, unsigned short* buf_len, unsigned short* to_here) {
	unsigned short ret = 0;
	if( *buf_len == *buf_pos || !isdigit(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isdigit(buf[*buf_pos]) ) {
		if( ret > 6553 || (ret == 6553 && buf[*buf_pos] > '5') )
			return 0;
		ret = ret*10 + (buf[*buf_pos]-'0');
		++*buf_pos;
	}
	*to_here = ret;
	return 1;
}
static unsigned char ishexdigit(char c) {
	return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' && c <= 'F');
}
static unsigned char read_hex(char* buf, unsigned short* buf_pos, unsigned short* buf_len, unsigned short* to_here) {
	unsigned short ret = 0;
	if( *buf_len == *buf_pos || !ishexdigit(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && ishexdigit(buf[*buf_pos]) ) {
		if( ret > 0xFFF )
			return 0;
		if( buf[*buf_pos] >= 'a' && buf[*buf_pos] <= 'f' ) {
			ret = ret*16 + 10 + (buf[*buf_pos]-'a');
		} else if( buf[*buf_pos] >= 'A' && buf[*buf_pos] <= 'F' ) {
			ret = ret*16 + 10 + (buf[*buf_pos]-'A');
		} else {
			ret = ret*16 + (buf[*buf_pos]-'0');
		}
		++*buf_pos;
	}
	*to_here = ret;
	return 1;
}

unsigned char infrared_logging, start_playback_automatically, dont_repeat_all;
unsigned char spectrum_attack, spectrum_decay, spectrum_min_brightness;
unsigned short agc_max_error, agc_delta_limit, agc_max, agc_filter_coeff;
extern unsigned char g_FFT_mode;

void reset_config_to_default() {
	g_FFT_mode = 0;
	start_playback_automatically = 1;
	dont_repeat_all = 0;
	playback_volume_set(100);
	Playback_Set_Order(sorted);
	spectrum_attack = 255;
	spectrum_decay = 255;
	spectrum_min_brightness = 8;
	agc_max_error = 0x0100;
	agc_delta_limit = 4;
	agc_filter_coeff = 16;
	agc_max = 0x4000;
}

static inline unsigned short abs(signed short val) {
	return val >= 0 ? val : -val;
}

unsigned char read_config(FIL* file) {
  char buf[256];
  unsigned char ret = 1;
  unsigned short buf_pos = 0, buf_len = 0;
  UINT read;

  while(1) {
    if( f_read(file, (BYTE*)buf + buf_len, sizeof(buf)-buf_len, &read) != FR_OK )
	  return 0;
    buf_len += read;
    if( buf_len == 0 )
      break;
	skip_space(buf, &buf_pos, &buf_len);
    if( buf_pos < buf_len ) {
	  if( buf[buf_pos] == '\n' ) {
        ++buf_pos;
      } else {
		if( this_token_is(buf, &buf_pos, &buf_len, "#") || this_token_is(buf, &buf_pos, &buf_len, "//") ) {
			// skip the comment
			while( buf_pos < buf_len && buf[buf_pos] != '\n' )
				++buf_pos;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "start") ) {
			if( next_token_is(buf, &buf_pos, &buf_len, "playback") ) {
				unsigned char auto_playback;
				if( !next_token_is(buf, &buf_pos, &buf_len, "automatically") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				auto_playback = get_next_token_boolean(buf, &buf_pos, &buf_len);
				if( auto_playback == 0xFF )
					return 0;
				start_playback_automatically = auto_playback;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "default") ) {
			if( next_token_is(buf, &buf_pos, &buf_len, "volume") ) {
				unsigned short volume = 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &volume) || volume > 100 )
					return 0;
				if( !this_token_is(buf, &buf_pos, &buf_len, "%") )
					return 0;

				playback_volume_set(volume);
			} else if( next_token_is(buf, &buf_pos, &buf_len, "repeat") ) {
				unsigned char repeat_all;
				if( !next_token_is(buf, &buf_pos, &buf_len, "all") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				repeat_all = get_next_token_boolean(buf, &buf_pos, &buf_len);
				if( repeat_all == 0xFF )
					return 0;

				dont_repeat_all = !repeat_all;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "playback") ) {
				playback_order_t order;
				if( !next_token_is(buf, &buf_pos, &buf_len, "order") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				if( next_token_is(buf, &buf_pos, &buf_len, "directory") )
					order = directory;
				else if( next_token_is(buf, &buf_pos, &buf_len, "sorted") )
					order = sorted;
				else if( next_token_is(buf, &buf_pos, &buf_len, "shuffle") )
					order = shuffle;
				else
					return 0;

				Playback_Set_Order(order);
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "mode") ) {
			unsigned short new_mode = 0;

			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &new_mode) || new_mode > 9 )
				return 0;
			g_FFT_mode = new_mode;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "spectrum") ) {
			if( next_token_is(buf, &buf_pos, &buf_len, "attack") ) {
				unsigned short attack = 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &attack) || attack > 255 )
					return 0;
				spectrum_attack = attack;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "decay") ) {
				unsigned short decay = 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &decay) || decay > 255 )
					return 0;
				spectrum_decay = decay;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "min") ) {
				unsigned short min_brightness = 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "brightness") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &min_brightness) || min_brightness > 255 )
					return 0;
				spectrum_min_brightness = min_brightness;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "agc") ) {
			if( next_token_is(buf, &buf_pos, &buf_len, "max") ) {
				if( next_token_is(buf, &buf_pos, &buf_len, "error") ) {
					if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
						return 0;
					skip_space(buf, &buf_pos, &buf_len);
					if( !read_integer(buf, &buf_pos, &buf_len, &agc_max_error) )
						return 0;
				} else if( next_token_is(buf, &buf_pos, &buf_len, "=") ) {
					skip_space(buf, &buf_pos, &buf_len);
					if( !read_integer(buf, &buf_pos, &buf_len, &agc_max) )
						return 0;
				} else {
					return 0;
				}
			} else if( next_token_is(buf, &buf_pos, &buf_len, "delta") ) {
				if( !next_token_is(buf, &buf_pos, &buf_len, "limit") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &agc_delta_limit) )
					return 0;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "filter") ) {
				if( !next_token_is(buf, &buf_pos, &buf_len, "coefficient") )
					return 0;
				if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
					return 0;
				skip_space(buf, &buf_pos, &buf_len);
				if( !read_integer(buf, &buf_pos, &buf_len, &agc_filter_coeff) )
					return 0;
			} else {
				return 0;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "remote") ) {
			unsigned char i, j;
			unsigned char which;
			unsigned short code = 0;

			if( !next_token_is(buf, &buf_pos, &buf_len, "code") )
				return 0;
			skip_space(buf, &buf_pos, &buf_len);
			for( i = 0; i < NUM_REMOTE_FNS; ++i ) {
				if( this_token_is(buf, &buf_pos, &buf_len, remote_codes[i].fn) )
					break;
			}
			if( i == NUM_REMOTE_FNS )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;

			memset(remote_codes[i].lower_codes, 0, sizeof(remote_codes[i].lower_codes));
			memset(remote_codes[i].upper_codes, 0, sizeof(remote_codes[i].upper_codes));

			j = 0;
			while( j < NUM_REMOTE_CODES ) {
				if( next_token_is(buf, &buf_pos, &buf_len, "NEC") ) {
					which = IR_NEC>>16;
			 	} else if( next_token_is(buf, &buf_pos, &buf_len, "RC5") ) {
					which = IR_RC5>>16;
				} else {
					return 0;
				}
				if( !next_token_is_spaceopt(buf, &buf_pos, &buf_len, "(") )
					return 0;
				if( next_token_is_spaceopt(buf, &buf_pos, &buf_len, "0x") ) {
					if( !read_hex(buf, &buf_pos, &buf_len, &code) )
						return 0;
				} else {
					if( !read_integer(buf, &buf_pos, &buf_len, &code) )
						return 0;
				}

				remote_codes[i].lower_codes[j] = code;
				remote_codes[i].upper_codes[j] = which;

				if( !next_token_is_spaceopt(buf, &buf_pos, &buf_len, ")") )
					return 0;
				if( ++j == NUM_REMOTE_CODES || !next_token_is_spaceopt(buf, &buf_pos, &buf_len, ",") )
					break;
			}
        } else if( this_token_is(buf, &buf_pos, &buf_len, "infrared") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "logging") )
				return 0;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			if( next_token_is(buf, &buf_pos, &buf_len, "on") ) {
				infrared_logging = 1;
			} else if( next_token_is(buf, &buf_pos, &buf_len, "off") ) {
				infrared_logging = 0;
			} else {
				return 0;
			}
        } else {
          return 0;
        }
	    if( !next_token_is(buf, &buf_pos, &buf_len, "\n") )
           return 0;
      }
    }
    memmove(buf, buf+buf_pos, buf_len-buf_pos);
    buf_len -= buf_pos;
    buf_pos = 0;
  }

  return ret;
}

unsigned char find_and_read_config() {
	FIL file;
	if( f_open(&file, "0:\\" CONFIG_FILENAME, FA_READ|FA_OPEN_EXISTING) == FR_OK ) {
		return read_config(&file);
	} else {
		return 1;
	}
}
